
from langchain_core.tools import tool
from pydantic import BaseModel, Field


from typing import Optional, Union
from datetime import datetime
#from opensearchpy import OpenSearch
from pdfminer.high_level import extract_text

import os


from ..core.SemanticIndexer.SemanticIndexer import SemanticIndexer
from ..core.shared.program_called_command_list import wait_safety





g_index = "base"
g_semantic_indexer = SemanticIndexer()
data=g_semantic_indexer.index_exists(g_index)
if not data:
    g_semantic_indexer.create_index(g_index)



def _set_long_memory_index(index):
    global g_index
    g_index = index

def _get_long_memory_index():
    global g_index
    return g_index


class RegisterTextInput(BaseModel):
    
    text: str = Field(..., description="登録するテキスト内容")
    summary: Optional[str] = Field(None, description="テキストの要約、ベクトル検索で使う（省略可能）")
    state: Optional[str] = Field(None, description="テキストを取得または作成した状態（省略可能）")
    purpose: Optional[str] = Field(None, description="テキストを取得または作成した目的（省略可能）")
    result: Optional[str] = Field(None, description="テキストを取得または作成した結果や作業の成否（省略可能）")
    section: Optional[str] = Field(None, description="セクション名（省略可能）")
    page: Optional[int] = Field(None, description="ページ番号（省略可能）")
@tool(args_schema = RegisterTextInput)
def register_text(text: str,
                  summary: Optional[str] = None,
                  state: Optional[str] = None,
                  purpose: Optional[str] = None,
                  result: Optional[str] = None,
                  section: Optional[str] = None, page: Optional[int] = None):
    """
    長期記憶用のデータ書き込み、登録機能です。
    テキストデータを書き込み、登録します。

    Args:
        text: 登録するテキスト内容
        summary: テキストの要約,ベクトル検索で使う（省略可能）
        state: テキストを取得または作成した状態（省略可能）
        purpose: テキストを取得または作成した目的（省略可能）
        result: テキストを取得または作成した結果や作業の成否（省略可能）
        section: セクション名（省略可能）
        page: ページ番号（省略可能）
    """
    print("register_text called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    g_semantic_indexer.register(index=g_index,
                                text=text,
                                summary=summary,
                                state=state,
                                purpose=purpose,
                                result=result,
                                section=section, page=page)

class RegisterPDFInput(BaseModel):
    
    filepath: str = Field(..., description="PDFファイルのパス")
    state: Optional[str] = Field(None, description="PDFファイルを取得または作成した状態、状況（省略可能）")
    purpose: Optional[str] = Field(None, description="PDFファイルを取得または作成した目的または記録する目的。（省略可能）")
    result: Optional[str] = Field(None, description="PDFファイルを取得または作成した結果や作業の成否（省略可能）")
    publication_date: Optional[Union[str, datetime]] = Field(None, description="公開日（省略可能）")
@tool(args_schema = RegisterPDFInput)
def register_pdf(filepath: str,
                 state: Optional[str] = None,
                 purpose: Optional[str] = None,
                 result: Optional[str] = None,
                 publication_date: Optional[Union[str, datetime]] = None):
    """
    長期記憶用のデータ書き込み、登録機能です。
    PDFファイルからテキストを抽出し、登録します。

    Args:
        filepath: PDFファイルのパス
        state: PDFファイルを取得または作成した状態、状況（省略可能）
        purpose: PDFファイルを取得または作成した目的。または記録する目的。（省略可能）
        result: PDFファイルを使用した作業の結果や作業の成否（省略可能）
        publication_date: 公開日（省略可能）
    """
    print("register_pdf called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要
    if os.path.exists(filepath):
        g_semantic_indexer.ingest_pdf(filepath=filepath, index=g_index,
                                  state=state,
                                  purpose=purpose,
                                  result=result,
                                  publication_date=publication_date)
        return "No Error"
    else:
        print("file not found\n")
        return "file not found"

class RegisterHTMLInput(BaseModel):
    
    filepath: str = Field(..., description="HTMLファイルのパス")
@tool(args_schema = RegisterHTMLInput)
def register_html(filepath: str):
    """
    長期記憶用のデータ書き込み、登録機能です。
    HTMLファイルからテキストを抽出し、登録します。

    Args:
        filepath: HTMLファイルのパス
    """
    print("register_html called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要
    if os.path.exists(filepath):
        g_semantic_indexer.ingest_html(filepath=filepath, index=g_index)
        return "No Error"
    else:
        print("file not found\n")
        return "file not found"


class RegisterImageInput(BaseModel):
    
    filepath: str = Field(..., description="画像ファイルのパス")
    text: Optional[str] = Field("", description="画像に関連するテキストの説明や、行った作業の内容（省略可能）")
    state: Optional[str] = Field(None, description="画像を取得または作成した状態、状況（省略可能）")
    purpose: Optional[str] = Field(None, description="画像を取得または作成した目的。または記録する目的。（省略可能）")
    result: Optional[str] = Field(None, description="画像を使用した作業の結果や作業の成否（省略可能）")
    section: Optional[str] = Field("image", description="セクション名（省略可能）")
    page: Optional[int] = Field(1, description="ページ番号（省略可能）")

@tool(args_schema = RegisterImageInput)
def register_image(filepath: str,
                   text: Optional[str] = "",
                   state: Optional[str] = None,
                   purpose: Optional[str] = None,
                   result: Optional[str] = None,
                   section: Optional[str] = "image", page: Optional[str] = 1):
    """
    長期記憶用のデータ書き込み、登録機能です。
    画像ファイルを指定されたインデックスに登録します。

    Args:
        
        filepath: 画像ファイルのパス
        text: 画像に関連するテキストの説明や、行った作業の内容（省略可能）
        state: 画像を取得または作成した状態、状況（省略可能）
        purpose: 画像を取得または作成した目的。または記録する目的。（省略可能）
        result: 画像を使用した作業の結果や作業の成否（省略可能）
        section: セクション名（省略可能）
        page: ページ番号（省略可能）
    """
    print("register_image called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要
    if os.path.exists(filepath):
        g_semantic_indexer.ingest_image(filepath=filepath, index=g_index, text=text,
                                    state=state,
                                    purpose=purpose,
                                    result=result,
                                    section=section, page=page)
        return "No Error"
    else:
        print("file not found\n")
        return "file not found"

class RegisterAudioInput(BaseModel):

    filepath: str = Field(..., description="音声ファイルのパス")
    text: Optional[str] = Field("", description="音声に関連するテキスト内容（省略可能）")
    state: Optional[str] = Field(None, description="音声を取得または作成した状態、状況（省略可能）")
    purpose: Optional[str] = Field(None, description="音声を取得または作成した目的。または記録する目的。（省略可能）")
    result: Optional[str] = Field(None, description="音声を使用した作業の結果や作業の成否（省略可能）")
    section: Optional[str] = Field("audio", description="セクション名（省略可能）")
    page: Optional[int] = Field(1, description="ページ番号（省略可能）")
@tool(args_schema = RegisterAudioInput)
def register_audio(filepath: str,
                   text: Optional[str] = "",
                   state: Optional[str] = None,
                   purpose: Optional[str] = None,
                   result: Optional[str] = None,
                   section: Optional[str] = "audio", page: int = 1):
    """
    長期記憶用のデータ書き込み、登録機能です。
    音声ファイルを登録します。 
    Args:
        index: インデックス名
        filepath: 音声ファイルのパス
        text: 音声に関連するテキスト内容（省略可能）
        state: 音声を取得または作成した状態、状況（省略可能）
        purpose: 音声を取得または作成した目的。または記録する目的。（省略可能）
        result: 音声を使用した作業の結果や作業の成否（省略可能）
        section: セクション名（省略可能）
        page: ページ番号（省略可能）
    """
    print("register_audio called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要
    if os.path.exists(filepath):
        g_semantic_indexer.ingest_audio(filepath=filepath, index=g_index, text=text,
                                    state=state,
                                    purpose=purpose,
                                    result=result,
                                    section=section, page=page)
        return "No Error"
    else:
        print("file not found\n")
        return "file not found"
class UpdateTextInput(BaseModel):
    
    id: str = Field(..., description="更新するドキュメントのID")
    new_text: str = Field(..., description="新しいテキスト内容")
    summary: Optional[str] = Field(None, description="テキストの要約、ベクトル検索で使う（省略可能）")
    section: Optional[str] = Field(None, description="新しいセクション名（省略可能）")
    page: Optional[int] = Field(None, description="新しいページ番号（省略可能）")
@tool(args_schema = UpdateTextInput)
def update_text(id: str, new_text: str, summary: Optional[str] = None, section: Optional[str] = None, page: Optional[int] = None):
    """
    長期記憶用のデータ更新、登録機能です。
    指定されたIDのドキュメントのテキストを更新します。

    Args:
        index: インデックス名
        id: 更新するドキュメントのID
        new_text: 新しいテキスト内容
        summary: テキストの要約、ベクトル検索で使う（省略可能）
        section: 新しいセクション名（省略可能）
        page: 新しいページ番号（省略可能）
    """
    print("update_text called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要
    g_semantic_indexer.update(index=g_index, id=id, new_text=new_text, section=section, page=page)

class DeleteDataInput(BaseModel):
    
    id: str = Field(..., description="削除するドキュメントのID")
@tool(args_schema = DeleteDataInput)
def delete_data(id: str):
    """
    長期記憶用のデータ削除機能です。
    指定されたIDのドキュメントを削除します。
    Args:
        index: インデックス名
        id: 削除するドキュメントのID
    """
    print("delete_data called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    g_semantic_indexer.delete(index=g_index, id=id)

class SearchKeywordInput(BaseModel):
    
    query: str = Field(..., description="検索クエリ")
    top_k: Optional[int] = Field(5, description="取得する上位件数")
@tool(args_schema = SearchKeywordInput)
def serch_keyword( query: str, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    キーワード検索を行います。

    Args:
        
        query: 検索クエリ
        top_k: 取得する上位件数

    Returns:
        検索結果のリスト
    """
    print("serch_keyword called\n", top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_keyword(index=g_index, query=query, top_k=top_k)

class SearchVectorInput(BaseModel):
    
    query_text: str = Field(..., description="検索クエリテキスト。内部で自動的にベクトル化され検索に利用される。")
    top_k: Optional[int] = Field(5, description="取得する上位件数")
@tool(args_schema = SearchVectorInput)
def search_vector( query_text: str, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    テキストのベクトル検索を行います。
    Args:
        
        query_text: 検索クエリテキスト。内部で自動的にベクトル化され検索に利用される。
        top_k: 取得する上位件数
    Returns:
        検索結果のリスト
    """
    print("search_vector called\n", top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_vector(index=g_index, query_text=query_text, top_k=top_k)
@tool(args_schema = SearchVectorInput)
def search_vector_state( query_text: str, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    検索クエリテキストは現在の状態を表すテキストを入れてください。
    状態テキストの内容の近いものを検索します。

    Args:
        
        query_text: 検索クエリテキスト。内部で自動的にベクトル化され検索に利用される。
        top_k: 取得する上位件数
    Returns:
        検索結果のリスト
    """
    print("search_vector_state called\n",query_text, top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_vector_state(index=g_index, query_text=query_text, top_k=top_k)

@tool(args_schema = SearchVectorInput)
def search_vector_purpose( query_text: str, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    検索クエリテキストは現在の目的テキストを入れてください。
    目的テキストの内容の近いものを検索します。
    Args:
        
        query_text: 検索クエリテキスト。内部で自動的にベクトル化され検索に利用される。
        top_k: 取得する上位件数
    Returns:
        検索結果のリスト
    """
    print("search_vector_purpose called\n", top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_vector_purpose(index=g_index, query_text=query_text, top_k=top_k)
@tool(args_schema = SearchVectorInput)
def search_vector_result( query_text: str, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    結果テキストの内容の近いものを検索します。
    Args:
        query_text: 検索クエリテキスト
        top_k: 取得する上位件数
    Returns:
        検索結果のリスト
    """
    print("search_vector_result called\n", top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_vector_result(index=g_index, query_text=query_text, top_k=top_k)


class SearchHybridInput(BaseModel):
    
    query_text: str = Field(..., description="検索クエリテキスト。内部で自動的にベクトル化され検索に利用される。")
    top_k: Optional[int] = Field(5, description="取得する上位件数")
@tool(args_schema = SearchHybridInput)
def search_hybrid( query_text: str, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    キーワード検索と意味検索のハイブリッド検索を行います。

    Args:
        query_text: 検索クエリテキスト。内部で自動的にベクトル化され検索に利用される。
        top_k: 取得する上位件数

    Returns:
        検索結果のリスト
    """
    print("search_hybrid called\n")
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_hybrid(index=g_index, query_text=query_text, top_k=top_k)


class SearchImageFileInput(BaseModel):
    file_name: str = Field(..., description="検索クエリ画像のファイル名")
    top_k: Optional[int] = Field(5, description="取得する上位件数")

@tool(args_schema = SearchImageFileInput)
def serch_image_file(file_name: str, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    指定されたファイル画像のベクトル検索を行います。
    
    :param file: 検索クエリ画像のファイル名
    :type file: str
    :param top_k: 取得する上位件数
    :type top_k: int
    """
    print("serch_image_file called\n", top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    if not os.path.exists(file_name):
        return  "ValueError: file not found"
    with open(file_name, "rb") as f:
        image_bytes = f.read()
    return g_semantic_indexer.search_image(index=g_index, image_bytes=image_bytes, top_k=top_k)
       
class SearchImageInput(BaseModel):
    
    image_bytes: bytes = Field(..., description="検索クエリ画像のバイトデータ")
    top_k: Optional[int] = Field(5, description="取得する上位件数")
@tool(args_schema = SearchImageInput)
def serch_image( image_bytes: bytes, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    画像のベクトル検索を行います。

    Args:
        
        image_bytes: 検索クエリ画像のバイトデータ
        top_k: 取得する上位件数

    Returns:
        検索結果のリスト
    """
    print("serch_image called\n", top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_image(index=g_index, image_bytes=image_bytes, top_k=top_k)

class SearchAudioInput(BaseModel):
    
    audio_bytes: bytes = Field(..., description="検索クエリ音声のバイトデータ")
    top_k: Optional[int] = Field(5, description="取得する上位件数")
@tool(args_schema = SearchAudioInput)
def search_audio(audio_bytes: bytes, top_k: int = 5):
    """
    長期記憶用のデータ検索、読み込み機能です。
    音声のベクトル検索を行います。 
    Args:
        
        audio_bytes: 検索クエリ音声のバイトデータ
        top_k: 取得する上位件数 
    Returns:
        検索結果のリスト
    """
    print("serch_audio called\n" ,top_k)
    global g_index
    wait_safety()  # リクエストの制限を越えないために必要    
    return g_semantic_indexer.search_audio(index=g_index, audio_bytes=audio_bytes, top_k=top_k)

def get_text_memory_register_tool():
    return [register_text]

def get_text_memory_search_tools():
    return [serch_keyword,
            search_vector,
            search_vector_state,
            search_vector_purpose,
            search_vector_result,
            search_hybrid]


def get_text_memory_tools_list():

    return [register_text,
            update_text,
            
            serch_keyword,
            search_vector,
            search_vector_state,
            search_vector_purpose,
            search_vector_result,
            search_hybrid]

def get_text_memory_edit_tools():
    return [update_text,
            delete_data,
            serch_keyword,
            search_vector,
            search_vector_state,
            search_vector_purpose,
            search_vector_result,
            search_hybrid]

def get_tools_list():
    return [
        register_text,
        register_pdf,
        #register_html,
        register_image,
        register_audio,
        
#        update_text,
#        delete_data,
        serch_keyword,
        search_vector,
        search_vector_state,
        search_vector_purpose,
        search_vector_result,
        search_hybrid,
        serch_image_file,
        serch_image,
        search_audio
    ]

def get_register_tools_list():
    return [
        register_text,

        register_pdf,
        register_image,
        register_audio,
    ]