
from typing import List, Optional, Dict
from datetime import datetime
import os


from ..core.FileList import FileList

from ..core.shared.program_called_command_list import wait_safety
from ..core.ProgressTracker import ProgressTracker

from ..core.shared import define

WORK_SPACE_DIR = define.WORK_SPACE_DIR
CURRENT_DIR= os.path.join(WORK_SPACE_DIR,"data/ProgressTracker")

    
#==============================
# グローバル変数
#==============================    
from langchain_core.tools import tool
from pydantic import BaseModel, Field

g_progress_tracker = None

class ProgressTrackerInput(BaseModel):
    title: str = Field(..., description="進捗管理のタイトル")
    purpose_of_creation: str = Field("", description="プロジェクト作成の目的")
@tool(args_schema = ProgressTrackerInput)
def create_progress_tracker(title: str, purpose_of_creation: str) -> None:
    """
    プロジェクトの進捗管理トラッカーを作成します。
    進捗管理トラッカーが各進捗管理ノードを管理します。
    ノード編集の前に実行してください。
    Args:
        :param title: 進捗管理のタイトル
        :type title: str
        :param purpose_of_creation: プロジェクト作成の目的
        :type purpose_of_creation: str
    """
    global g_progress_tracker
    wait_safety()
    g_progress_tracker = ProgressTracker(title, purpose_of_creation)

    # ファイルリストのパスの設定
    file_path = os.path.join(CURRENT_DIR, "progress_file_list.json")
    file_list=FileList()
    file_list.set_data_file_path(file_path)
    if not os.path.exists(CURRENT_DIR):
        file_list.create_folder(CURRENT_DIR)
    if not os.path.exists(file_path):
        file_list.list_files_recursive(CURRENT_DIR)
        
    file_list.set_file_description(file_path, purpose_of_creation, "進捗管理データファイル")
    file_list.save_file_descriptions()

class SetProgressRootInput(BaseModel):
    name: str = Field(..., description="ルートノードの名前")
    description: str = Field("", description="ルートノードの説明")
    status: str = Field("未着手", description="ルートノードのステータス")
    start_time: Optional[datetime] = Field(None, description="ルートノードの開始時刻")

@tool(args_schema = SetProgressRootInput)    
def set_progress_root(name: str, description: str = "", status: str = "未着手",
                       start_time: Optional[datetime] = None) -> str:
    """
    プロジェクトの進捗管理のルートノードを作成し、設定します。
    Docstring for set_progress_root
    Args:
        :param name: ルードノードの名前
        :type name: str
        :param description: ルートノードの説明
        :type description: str
        :param status: ルートノードのステータス
        :type status: str
        :param start_time: ルートノードの開始時間
        :type start_time: Optional[datetime]
    Return: 作成されたルートノードのID
        :rtype: str
    """
    # ファイルリストのパスの設定
    #
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    root_node = ProgressNode(name=name, description=description, status=status, start_time=start_time)
    g_progress_tracker.set_root(root_node)
    return root_node.id


class AddProgressTaskInput(BaseModel):
    parent_id: str = Field(..., description="親タスクノードのID")
    name: str = Field(..., description="新たに追加するタスクノードの名前")
    description: str = Field("", description=" 新たに追加するタスクの内容")
    can_run_parallel: bool = Field(False, description="並列実行可能かどうか")
@tool(args_schema = AddProgressTaskInput)
def add_progress_task(parent_id: str, name: str, description: str = "", can_run_parallel: bool = False) -> str:
    """
    プロジェクトの進捗管理の親ノードを指定してタスクを追加（追加されたタスクのIDを返す）
    Args:
        :param parent_id: 親となるタスクノードのID
        :type parent_id: str
        :param name: 新たに追加するタスクノードの名前
        :type name: str
        :param description: 新たに追加するタスクの内容, defaults to ""
        :type description: str, optional
        :param can_run_parallel: 並列事項可能かどうか, defaults to False
        :type can_run_parallel: bool, optional
    Return: 
        追加されたタスクのID
        
        :rtype: str
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    print("add_progress_task:", parent_id, name, description, can_run_parallel)
    new_task = ProgressNode(name=name, description=description, can_run_parallel=can_run_parallel)
    success = g_progress_tracker.add_task(parent_id, new_task)
    if not success:
        raise ValueError(f"Parent node with ID {parent_id} not found.")
    return new_task.id

class UpdateProgressStatusInput(BaseModel):
    node_id: str = Field(..., description="更新するタスクノードのID")
    new_status: str = Field(..., description="新しいステータス")
    note: str = Field(..., description="新しいメモ書き")
@tool(args_schema = UpdateProgressStatusInput)
def update_progress_status(node_id: str, new_status: str, note: str) -> bool:
    """
    プロジェクトの進捗管理のタスクノードのIDを指定してステータスを更新（成功すればTrue、失敗すればFalse）
    ステータス：未着手、作業中、完了など
    Args:
        :param node_id: 更新するタスクノードのID
        :type node_id: str
        :param new_status: タスクノードの新しいステータス
        :type new_status: str
    Return:
         成功すればTrue、失敗すればFalse
        :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    return g_progress_tracker.update_status_by_id(node_id, new_status, note)

class UpdateProgressNoteInput(BaseModel):
    node_id: str = Field(..., description="タスクノードのID")
    note: str = Field(..., description="新しいメモ書き")
@tool(args_schema = UpdateProgressNoteInput)
def update_progress_note(node_id: str, note: str) -> bool:
    """
    プロジェクトの進捗管理のタスクノードのIDを指定してメモ書きを更新（成功すればTrue、失敗すればFalse）
    Args:
        :param node_id: タスクノードのID
        :type node_id: str
        :param note: メモ書き
        :type note: str
    Return:
         成功すればTrue、失敗すればFalse

        :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    return g_progress_tracker.updat_note_by_id(node_id, note)
    

class RemoveProgressTaskInput(BaseModel):
    node_id: str = Field(..., description="削除するノードのID")
@tool(args_schema = RemoveProgressTaskInput)
def remove_progress_task(node_id: str) -> bool:
    """
    プロジェクトの進捗管理のノードをIDで削除（成功すればTrue、失敗すればFalse）
    Args:
        :param node_id: Description
        :type node_id: str
    Return: 
        成功すればTrue、失敗すればFalse
        :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    return g_progress_tracker.remove_task(node_id)

@tool
def get_progress_mermaid() -> str:
    """
    プロジェクトの進捗管理のMermaid記法でガントチャートを出力（人間向け可視化）

    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    return g_progress_tracker.to_mermaid()

@tool
def get_progress_json() -> Dict:
    """
    プロジェクトの進捗管理全体をJSON形式で出力（AIや外部連携用）

    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    return g_progress_tracker.to_json()



def _save_progress_to_file() -> None:
    """
    プロジェクトの進捗管理の進捗データをJSONファイルに保存
  

    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    file_dir= CURRENT_DIR
    _create_folder(file_dir)
    g_progress_tracker.save_to_file(file_dir)

@tool
def save_progress_to_file() -> None:
    """
    プロジェクトの進捗管理の進捗データをJSONファイルに保存

    """
    wait_safety()    
    _save_progress_to_file()

def _load_progress_from_file(title: str = "") -> None:
    """
    JSONファイルからプロジェクトの進捗管理の進捗データを読み込み

    """   
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    file_dir=CURRENT_DIR
    _create_folder(file_dir)
    
    g_progress_tracker.load_from_file(file_dir, title=g_progress_tracker.title)

class LoadProgressFromFileInput(BaseModel):
    title: str = Field("", description="進捗管理のタイトル")
@tool(args_schema = LoadProgressFromFileInput)
def load_progress_from_file(title: str = "") -> None:
    """
    JSONファイルからプロジェクトの進捗管理の進捗データを読み込み
    ノード編集の前に実行してください。
    Args:
        :param title: 進捗管理のタイトル
        :type title: str
    """   
    wait_safety()    
    _load_progress_from_file(title=title)



class SetProgressNodeWorkParentsInput(BaseModel):
    node_id: str = Field(..., description="ノードのID")
    parent_ids: List[str] = Field(..., description="親ノードのIDリスト")
@tool(args_schema = SetProgressNodeWorkParentsInput)
def set_progress_node_work_parents(node_id: str, parent_ids: List[str]) -> bool:
    """
    プロジェクトの進捗管理のノードに親ノードIDリストを設定します。
    Args:
        :param node_id: ノードのID
        :type node_id: str
        :param parent_ids: 親ノードのIDリスト
        :type parent_ids: List[str]
    Return:
       成功すればTrue、失敗すればFalse
       :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    node = g_progress_tracker.find_node_by_id(node_id)
    if node:
        node.work_parent_ids = parent_ids
        return True
    return False

class GetProgressNodeWorkParentsInput(BaseModel):
    node_id: str = Field(..., description="ノードのID")
@tool(args_schema = GetProgressNodeWorkParentsInput)
def get_progress_node_work_parents(node_id: str) -> Optional[List[str]]:
    """
    プロジェクトの進捗管理のノードの親ノードIDリストを取得します。
    Args:
        :param node_id: ノードのID
        :type node_id: str
    Return:
        親ノードのIDリスト、ノードが見つからなければNone
        :rtype: Optional[List[str]]
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    node = g_progress_tracker.find_node_by_id(node_id)
    if node:
        return node.work_parent_ids
    return None

class AddProgressNodeWorkParentInput(BaseModel):
    node_id: str = Field(..., description="ノードのID")
    parent_id: str = Field(..., description="親ノードのID")
@tool(args_schema = AddProgressNodeWorkParentInput)
def add_progress_node_work_parent(node_id: str, parent_id: str) -> bool:
    """
    プロジェクトの進捗管理のノードに親ノードIDを追加します。
    Args:
        :param node_id: ノードのID
        :type node_id: str
        :param parent_id: 親ノードのID
        :type parent_id: str
    Return:
         成功すればTrue、失敗すればFalse
        :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    node = g_progress_tracker.find_node_by_id(node_id)
    if node:
        node.add_parent_id(parent_id)
        return True
    return False
class RemoveProgressNodeWorkParentInput(BaseModel):
    node_id: str = Field(..., description="ノードのID")
    parent_id: str = Field(..., description="親ノードのID")
@tool(args_schema = RemoveProgressNodeWorkParentInput)
def remove_progress_node_work_parent(node_id: str, parent_id: str) -> bool:
    """
    プロジェクトの進捗管理のノードから親ノードIDを削除します。
    Args;
        :param node_id: ノードのID
        :type node_id: str
        :param parent_id: 親ノードのID
        :type parent_id: str
    Return:
        成功すればTrue、失敗すればFalse
        :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    node = g_progress_tracker.find_node_by_id(node_id)
    if node:
        node.remove_parent_id(parent_id)
        return True
    return False


class UnassignProgressNodeMemberInput(BaseModel):
    node_id: str = Field(..., description="ノードのID")
    member_name: str = Field(..., description="メンバー名")
@tool(args_schema = UnassignProgressNodeMemberInput)
def unassign_progress_node_member(node_id: str, member_name: str) -> bool:
    """
    プロジェクトの進捗管理のノードからメンバーの割り当てを解除します。
    Args:
        :param node_id: ノードのID
        :type node_id: str
        :param member_name: メンバー名
        :type member_name: str
    Return:
        成功すればTrue、失敗すればFalse
       :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    node = g_progress_tracker.find_node_by_id(node_id)
    if node:
        node.unassign_member(member_name)
        return True
    return False
    

class AssignProgressNodeMemberInput(BaseModel):
    node_id: str = Field(..., description="ノードのID")
    member_name: str = Field(..., description="メンバー名")
@tool(args_schema = AssignProgressNodeMemberInput)
def assign_progress_node_member(node_id: str, member_name: str) -> bool:
    """
    プロジェクトの進捗管理のノードにメンバーを割り当てます。
    Args:
        :param node_id: ノードのID
        :type node_id: str
        :param member_name: メンバー名
        :type member_name: str
    Return:
        成功すればTrue、失敗すればFalse
        :rtype: bool
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    node = g_progress_tracker.find_node_by_id(node_id)
    if node:
        node.assign_member(member_name)
        return True
    return False

class GetProgressNodeMembersInput(BaseModel):
    node_id: str = Field(..., description="ノードのID")
@tool(args_schema = GetProgressNodeMembersInput)
def get_progress_node_members(node_id: str) -> Optional[List[str]]:
    """
    進捗管理のノードに割り当てられたメンバーリストを取得します。
    Args:
        :param node_id: ノードのID
        :type node_id: str
    Returns:
        メンバーリスト、ノードが見つからなければNone
        :rtype: Optional[List[str]]
    """
    global g_progress_tracker
    wait_safety()    
    if g_progress_tracker is None:
        raise ValueError("ProgressTracker is not initialized.")
    node = g_progress_tracker.find_node_by_id(node_id)
    if node:
        return node.get_members()
    return None

#==============================
# 進捗管理トラッカーのファイルリスト管理ツール
def _get_progress_file_descriptions() -> str:
    """
    プロジェクトの進捗管理トラッカーのデータの保存先フォルダ内のファイルリストを取得します。

    :return: ファイルパスのリスト
    :rtype: List[str]
    """
    return file_list._get_and_create_all_file_descriptions(CURRENT_DIR)


@tool
def get_progress_file_list() -> str:
    """
    プロジェクトの進捗管理トラッカーのデータの保存先フォルダ内のファイルリストを取得します。

    :return: ファイルパスのリスト
    :rtype: List[str]
    """
    wait_safety()    
    return _get_progress_file_descriptions()

class SetProgressFileDescriptionInput(BaseModel):
    file_path: str = Field(..., description="プロジェクトの進捗管理トラッカーのファイルのパス")
    description: str = Field(..., description="ファイルの説明")
    note: str = Field("", description="ファイルのメモ書き")
@tool(args_schema = SetProgressFileDescriptionInput)
def set_progress_file_description(file_path: str, description: str, note: str="") -> None:
    """
    プロジェクトの進捗管理トラッカーのデータの保存先フォルダ内のファイルリストにファイルの説明を追加します。
    この機能呼び出しの前にget_progress_file_listを実行してファイルリストを初期化してください。
    Args:
        :param file_path: ファイルのパス
        :type file_path: str
        :param description: ファイルの説明
        :type description: str
        :param note: ファイルのメモ書き
        :type note: str
    """
    wait_safety()    
    file_list._set_file_description(file_path, description, note)


def _save_progress_file_descriptions() -> None:
    file_list._save_file_descriptions()
@tool
def save_progress_file_descriptions() -> None:
    """
    プロジェクトの進捗管理トラッカーのデータの保存先フォルダ内の
    ファイルの説明を保存します。
    この機能呼び出しの前にget_progress_file_listを実行してファイルリストを初期化してください。
    """
    wait_safety()    
    file_list._save_file_descriptions()


def get_tools_list():
    return [
        create_progress_tracker,
        set_progress_root,
        add_progress_task,
        update_progress_status,
        update_progress_note,
        remove_progress_task,
        get_progress_mermaid,
        get_progress_json,
        save_progress_to_file,
        load_progress_from_file,
        set_progress_node_work_parents,
        get_progress_node_work_parents,
        add_progress_node_work_parent,
        remove_progress_node_work_parent,
        assign_progress_node_member,
        unassign_progress_node_member,
        get_progress_node_members,
        get_progress_file_list,
        set_progress_file_description,
        save_progress_file_descriptions
    ]