#!/usr/bin/env python3
"""
ファイル: list_files_recursive.py
作成者: ChatGPT
変更修正者:robotcreation-dialy.com

説明:
指定されたフォルダに含まれるすべてのファイルパス（サブフォルダも含む）のリストを返す単一の関数 `list_files_recursive(folder_path)` を提供します。返されるパスは絶対パスで、アルファベット順に並べられています。
スクリプトを直接実行した際に、現在の作業ディレクトリで関数を実行する小さなテストハーネスが含まれています。
"""


from langchain_core.tools import tool
from pydantic import BaseModel, Field

from ..core.FileList import FileList
g_file_list = FileList()



# --------------------------------------------------------------------------- #
# Test harness – runs when the script is executed directly
# --------------------------------------------------------------------------- #
class SetFileDescription(BaseModel):
    file_path: str = Field()
    description: str = Field(default="")

@tool(args_schema=SetFileDescription)
def set_file_description(file_path, purpose_of_creation = "",description=""):
    """
    指定されたファイルの説明を作成します。

    Args:
      file_path: 説明を作成するファイルのパス。
      purpose_of_creation: ファイル作成目的
      description: ファイル内容の説明
    """
    g_file_list.set_file_description(file_path, purpose_of_creation,description)

@tool
def save_file_descriptions():
    """
    ファイルの説明を指定されたパスに保存します。

    Args:
      output_path: 説明を保存するファイルのパス。
    """
    g_file_list.save_file_descriptions()


@tool
def load_file_descriptions():
    """
    指定されたパスからファイルの説明を読み込みます。

    Args:
      input_path: 説明を読み込むファイルのパス。
    """
    g_file_list.load_file_descriptions()


def _get_file_description(file_path):
    """
    指定されたファイルの説明を取得します。

    Args:
      file_path: 説明を取得するファイルのパス。

    Returns:
      ファイルの説明。説明が存在しない場合は空文字列を返す。
    """
    global g_file_list
    return g_file_list.get_file_description(file_path, "")



@tool
def get_all_file_descriptions() -> str:
    """
    現在のすべてのファイルの説明を取得します。

    Returns:
      すべてのファイルの説明を含む辞書。
    """
    return g_file_list.get_all_file_descriptions()


if __name__ == "__main__":
    import sys

    # Use the directory from which the script is run if no argument is given
    target_dir = g_file_list.get_directory(__file__)

    try:
        files = g_file_list.list_files_recursive(target_dir)
        print(f"Found {len(files)} file(s) under '{target_dir}':")
        target_dir = target_dir.replace("/", "\\")
        print(target_dir)
        for f in files:
            f = f.replace(target_dir, "")
            
            print(f)
        print(target_dir)
    except ValueError as e:
        print(f"Error: {e}")
        sys.exit(1)
