
import sqlite3
from typing import List, Tuple


class ThreadedBBS:
    def __init__(self, db_path: str = "bbs.db"):
        self.db_path = db_path
        self._init_db()

    def _init_db(self):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        # スレッドテーブル
        c.execute("""
            CREATE TABLE IF NOT EXISTS threads (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT,
                purpose TEXT
            )
        """)

        # 投稿テーブル（スレッドに紐付け）
        c.execute("""
            CREATE TABLE IF NOT EXISTS posts (
                id INTEGER PRIMARY KEY AUTOINCREMENT,   -- 全体ユニークID
                thread_id INTEGER,
                seq INTEGER,                            -- スレッド内番号
                name TEXT,
                message TEXT,
                FOREIGN KEY(thread_id) REFERENCES threads(id)
            )
        """)

        conn.commit()
        conn.close()

    # --- スレッド管理 ---
    def create_thread(self, title: str, purpose: str = "") -> int:
        """新しいスレッドを作成してIDを返す"""
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute("INSERT INTO threads (title, purpose) VALUES (?, ?)", (title, purpose))
        conn.commit()
        tid = c.lastrowid
        conn.close()
        return tid


    def list_threads(self) -> List[Tuple[int, str, str]]:
        """スレッド一覧を取得 (id, title, purpose)"""
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute("SELECT * FROM threads ORDER BY id ASC")
        threads = c.fetchall()
        conn.close()
        return threads

    # --- 投稿管理 ---
    def add_post(self, thread_id: int, name: str, message: str) -> None:
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        # スレッド内の最大seqを取得して +1
        c.execute("SELECT COALESCE(MAX(seq), 0) + 1 FROM posts WHERE thread_id=?", (thread_id,))
        seq = c.fetchone()[0]
        c.execute("INSERT INTO posts (thread_id, seq, name, message) VALUES (?, ?, ?, ?)",
                  (thread_id, seq, name, message))
        conn.commit()
        conn.close()


    def list_posts(self, thread_id: int, limit: int = 10) -> List[Tuple[int, int, str, str]]:
        """指定スレッドの投稿一覧を取得（古い順）"""
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute("SELECT * FROM posts WHERE thread_id=? ORDER BY id ASC LIMIT ?", 
                  (thread_id, limit))
        posts = c.fetchall()
        conn.close()
        return posts

