
"""
Docstring for PythonExecutor
pythonファイルを実行するクラスです。
"""

import os
import sys
import subprocess
import ast

import threading
import time
import msvcrt
import threading
import queue


from ..shared import define
from .SafetyChecker import check_code_safety
from ..shared.program_called_command_list import load_python_file
WORK_SPACE_DIR = define.WORK_SPACE_DIR
AGENT_SPACE_DIR = define.AGENT_SPACE_DIR
CODE_SPACE_DIR = define.CODE_SPACE_DIR
DELOVERABLES_SPACE_DIR = define.DELOVERABLES_SPACE_DIR
TEMP_SPACE_DIR = define.TEMP_SPACE_DIR
PYTHON_PATH =  define.PYTHON_PATH



class PythonFileExecuteData():
    """pythonファイル実行時の出力保存"""
    def __init__(self):
        self.clear()

    def clear(self):
        self.stdout = ""
        self.error = ""
        self.returncode = 0


class PythonExecutor():
    def __init__(self):
        self.file_name=""
        self.data=""
        self.result_data = PythonFileExecuteData()
        self.running_time = 0
        self.exitFlag = False
        self.output_queue = queue.Queue()
        self.error_queue = queue.Queue()
        self.all_output = ""
        self.all_error = ""
        self.run_thread=None
        self.run_process=None
        self.safety_level = "middle"
        self.process_running = False


    def read_output(self, process):
        while True:
            line = process.stdout.readline()
            if line == b'':
                break

            try:
                print(line.decode('utf-8'), end='')
                self.result_data.stdout += line.decode('utf-8')
            except UnicodeDecodeError:
                if str is type(line):
                    self.result_data.error += line


    def read_error(self, process):
        while True:
            line = process.stderr.readline()
            if line == b'':
                break
            try:
                print(line.decode('utf-8'), end='')
                self.result_data.error += line.decode('utf-8')

            except UnicodeDecodeError:
                if str is type(line):
                    self.result_data.error += line




    def run_code_in_thread(self, code, input="", timeout=0):
        self.process_running = True
        if self.run_process is not None:
            if None is self.run_process.poll():
                return 
        errors=self.check_code(code)
        if 0 != len(errors):
            return 
        process = subprocess.Popen(
            ["python3", "-c", "import sys; exec(sys.stdin.read())", input],
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            bufsize=1  # 行単位で読み取りやすくする
        )
        self.run_process = process
        sttime = time.time()
        self.running_time = 0
        # コード送信
        process.stdin.write(code)
        process.stdin.close()
        self.exitFlag = False
        # メインスレッドでキー入力待ち
        while True:
            # キー入力があるか確認
            try:

                if msvcrt.kbhit():
                    # キー入力があれば、子プロセスにデータを送信
                    input_data = msvcrt.getch()

                    if input_data == b'\x1b':  # escキー
                        print("escキーが押されました")
                        process.terminate()
                        break

                    # キー入力があったら、子プロセスにデータを送信
                    if process.poll() is None:
                        if process.stdin is not None:
                            process.stdin.write(input_data)
                            process.stdin.flush()
                        else:
                            print("子プロセスは標準入力を受け取っていません")
                    else:
                        print("子プロセスはすでに終了しています")
                        break
                else:
                    time.sleep(0.01)
                    if 0 < len(self.result_data.error):
                        print("エラーが発生しました。")
                        process.terminate()
                        break

                # stdout を逐次読み取って queue に入れる
                for line in process.stdout:
                    
                    self.output_queue.put(line)
                
                for line in process.stderr:
                    
                    self.error_queue.put(line)
                self.running_time = time.time() - sttime

                if 0 < timeout:
                    if (timeout < time.time() - sttime):
                        process.terminate()
                        print(str(timeout)+"秒正常に動作しました。")

                        break

                if self.exitFlag:
                    process.terminate()
                    break
                if None is self.run_process.poll():
                    break


            except EOFError:
                break


        # stdout を逐次読み取って queue に入れる

        for line in process.stdout:
            print("line",line)
            self.output_queue.put(line)
        for line in process.stderr:
            self.error_queue.put(line)
        process.stdout.close()
        process.wait()
        self.process_running = False


    def execute_data(self, data, input="", timeout=0):
        """
        別プロセスで実行される関数。
        # 別のPythonコードを実行する
        # python_fileのコードを実行します。input_dataは引数です。
        Args:
            python_file: 実行したいパイソンファイル
            input_data: pythonファイルに渡す引数
        Returns:
            実行結果と標準出力。エラーが発生した場合は、エラーメッセージと標準出力を返します。
        """
        print("python_data_execute:")
        self.process_running = True
        thread = threading.Thread(
            target=self.run_code_in_thread,
            args=(data,input,timeout)
        )
        thread.start()



    def execute_file(self, file_name, input_data="", timeout=0):
        """
        
        # 別のPythonコードを実行する
        # python_fileのコードを実行します。input_dataは引数です。
        Args:
            python_file: 実行したいパイソンファイル
            input_data: pythonファイルに渡す引数
        Returns:
            実行結果と標準出力。エラーが発生した場合は、エラーメッセージと標準出力を返します。
        """
        print("python_file_execute:")
        self.process_running = True
        if self.run_process is not None:
            if None is self.run_process.poll():
                return 
            
        code=load_python_file(file_name)
        
        errors=self.check_code(code)
        if 0 != len(errors):
            return 
        self.file_name=file_name
        self.data=code
        self.result_data.clear()

        thread = threading.Thread(
            target=self.run_code_in_thread,
            args=(self.data,input_data,timeout)
        )
        thread.start()


    def check_error_python_file_execute(self, python_file,
                                        input_data, normal_termination_time=5):
        """
        別プロセスで実行される関数。
        # 別のPythonコードを実行する
        # python_fileのコードを実行します。input_dataは引数です。
        Args:
            python_file: 実行したいパイソンファイル
            input_data: pythonファイルに渡す引数
        Returns:
            実行結果と標準出力。エラーが発生した場合は、エラーメッセージと標準出力を返します。
        """
        print("check_error_python_file_execute:")
        self.process_running = True
        self.result_data.clear()
        self.run_execute_file_thread(python_file, input_data, normal_termination_time)
        
        return self.get_all_output(), self.get_all_error(), ""
    
    def get_output(self):
        result = ""

        while not self.output_queue.empty():
            
            result+=str(self.output_queue.get())
        self.all_output += result
        return result

    def get_error(self):
        result = ""

        while not self.error_queue.empty():
            result+=str(self.error_queue.get())
        self.all_error += result    
        return result
    
    def get_all_output(self):
        result = ""

        while not self.output_queue.empty():
            
            result+=str(self.output_queue.get())
        
        self.all_output += result        
        return self.all_output
    
    def get_all_error(self):
        result = ""

        while not self.error_queue.empty():
            result+=str(self.error_queue.get())
        self.all_error += result    
        return self.all_error
    

    def get_running_time(self):
        return self.running_time
    
    def is_running_process(self):
        return self.process_running

    
    def exit(self):
        self.exitFlag = True
        time.sleep(0.1)
        if self.run_process is not None:
            if None is self.run_process.poll():
                self.run_process.terminate()
        self.process_running = False
       
    def check_code(self,
                   code: str,
                   level: str = "middle",
                   large_number_threshold: int = 10**7,
                    ) -> list[str]:
        self.safety_level = level
        errors = check_code_safety(code, self.safety_level, large_number_threshold)
        if 0 != len(errors):
            self.all_error = "\r\n".join(errors)
        return errors
    
    def set_safety_level(self, level: str):
        self.safety_level = level
    
    def get_safety_level(self):
        return self.safety_level
    
    def wait_process_end(self):
        while self.process_running:
            time.sleep(0.1)
