import os
from PySide6.QtWidgets import QComboBox, QHBoxLayout,QWidget, QLabel
from PySide6.QtCore import Qt
from PySide6.QtGui import QColor                             
from PySide6.QtWidgets import QPushButton,QPushButton, QComboBox
import requests
import json
from PySide6.QtWidgets import QApplication

import time
import inspect
class SortedComboBox(QComboBox):
    def __init__(self, *args, sort_column=0, sort_order=Qt.AscendingOrder, **kwargs):
        super().__init__(*args, **kwargs)
        self.sort_column = sort_column
        self.sort_order = sort_order
        self.sort_function = None

    def showPopup(self):
        self.model().sort(self.sort_column, self.sort_order)
        if self.sort_function:
            self.sort_function(self)
        super().showPopup()

    def set_sort_function(self, func):
        self.sort_function = func



    def _item_text_to_key(self, text: str) -> tuple[str, str]:
        """「model_provider:name」 → (provider, name) のタプルを返す"""
        if ":" not in text:
            return ("", "")
        provider, name = text.split(":", maxsplit=1)
        return (provider.strip(), name.strip())

    # --------------------------------------------------------------
    # ★★ 重要な差分：新規キーの追加ロジックを **削除** し、元の順序を保持
    # --------------------------------------------------------------
    def sort_by_importance(self, sorted_models) -> None:
        """
        Combo にすでに存在する項目だけを重要度順に並べ替える。
        新しいキーは一切追加せず、Combo の現在構成はそのまま維持します。
        """
        # 1️⃣ 現行の文字列 → (model_provider, model_name) マッピング
        
        current_items: list[str] = []
        index_to_key: dict[int, tuple[str, str]] = {}
        key_to_index: dict[tuple[str, str], int] = {}

        for idx in range(self.count()):
            txt = self.itemText(idx)
            current_items.append(txt)
            #print("txt",txt)
            provider, name= self._item_text_to_key(txt)   # (provider, name) のタプル
            key = (provider, name)
            #print("key",key)
            if key:
                index_to_key[idx] = key
                key_to_index[key] = idx

        # 2️⃣ 集計結果取得
        sorted_aggs = sorted_models
        
        #print("sorted_aggs", sorted_aggs)
        # 3️⃣ Combo に存在するキーだけを重要度降順に抽出・ソート
        #print("key_to_index", key_to_index)

        present_keys_in_order = [
            agg for agg in sorted_aggs
            if (agg["model_provider"] ,agg["model_name"]) in key_to_index
        ]
        
        present_keys_in_order.sort(
            key=lambda a: (-a["importance_sum"],
                           key_to_index[(a["model_provider"], a["model_name"])])
        )
        

        # 4️⃣ 新しい順序（インデックスリスト）を作成
        new_order_indices: list[int] = []
        #print("present_keys_in_order",present_keys_in_order)
        # (a) 重要度が高い順に「Combo にある」キーだけを配置
        for agg in present_keys_in_order:
            idx = key_to_index[(agg["model_provider"], agg["model_name"])]
            new_order_indices.append(idx)
        
        # (b) ソート対象外だった残りの項目は、元の相対順序を保ったまま最後に続く
        for old_idx in range(self.count()):
            if old_idx not in new_order_indices:
                new_order_indices.append(old_idx)

        self.blockSignals(True)

        # 5️⃣ 再構築（既存項目のみ入れ替える）
        self.clear()
        for idx in new_order_indices:
            name = current_items[idx]
            self.addItem(name)
            new_index = self.count() - 1

            # ★ ここで色付けする（再構築後）
            lower = name.lower()
            if "ollama" in lower:
                self.setItemData(new_index, QColor("#FFF0F0"), Qt.BackgroundRole)
            elif "lmstudio" in lower:
                self.setItemData(new_index, QColor("#F0F0FF"), Qt.BackgroundRole)


        self.blockSignals(False)

        self.setCurrentIndex(0)          # 必要なら適宜調整
        

    # --------------------------------------------------------------


    # 省略：_ensure_model_registry_table、_calc_importance、get_sorted_models などはそのまま使用可


class ModelSelectWidget(QWidget):
    def __init__(self):


        super().__init__()
        self.model_label = QLabel("Select Model:")
        self.tools_vision = QLabel("")
          # モデル選択コンボボックス
        self.model_combo = SortedComboBox()
        self.refresh_button = QPushButton("Refresh Models")
        self.refresh_button.clicked.connect(self.refresh_model_list)

        self.settings_layout = QHBoxLayout()

        #self.refresh_model_list()
        self.settings_layout.addStretch(1)
        self.settings_layout.addWidget( self.model_label)
        self.settings_layout.addWidget(self.model_combo)
        self.settings_layout.addWidget(self.tools_vision)
        self.settings_layout.addStretch(1)
        self.refresh_model_list()
        

        self.setLayout(self.settings_layout)


        self.model_combo.activated.connect(self.on_model_selected)

        self.selected_function = None

    def refresh_model_list(self):

        self.available_models = self.get_available_llm_models()
        model_names = [name for name in self.available_models]
        
        self.model_combo.blockSignals(True)
        self.model_combo.clear()


        for name in model_names:
            self.model_combo.addItem(name)
            index = self.model_combo.count() - 1
        
            if "ollama" in name.lower():
                self.model_combo.setItemData(index, QColor("#FFF0F0"), Qt.BackgroundRole)
            elif "lmstudio" in name.lower():
                self.model_combo.setItemData(index, QColor("#F0F0FF"), Qt.BackgroundRole)
        self.model_combo.blockSignals(False)
        
    def get_available_llm_models(self) -> list[str]:
        # 書式: "provider:model_identifier"
        static_models = []
        if None is not os.getenv("GOOGLE_API_KEY"):

            static_models = [
                "google_genai:gemini-2.5-flash",
        #        "google_genai:gemini-1.5-pro-latest", # Vision対応の可能性
                # "hf:microsoft/Phi-3-mini-4k-instruct" # HuggingFaceモデルの例 (別途HuggingFaceLLMの実装が必要)
            ]
        ollama_models = []
        try:
            response = requests.get("http://localhost:11434/api/tags", timeout=3) # タイムアウトを3秒に設定
            response.raise_for_status() # HTTPエラーがあれば例外を発生
            models_data = response.json()
            if "models" in models_data and isinstance(models_data["models"], list):
                for model_info in models_data["models"]:
                    if "name" in model_info:
                        ollama_models.append(f"ollama:{model_info['name']}")
            if not ollama_models:
                print("Warning: No models found in Ollama API response, though connection was successful.")
        except requests.exceptions.Timeout:
            print("Warning: Timeout when trying to connect to Ollama to get model list.")
        except requests.exceptions.ConnectionError:
            print("Warning: Could not connect to Ollama server (e.g., server not running).")
        except requests.exceptions.RequestException as e:
            print(f"Warning: Error connecting to Ollama to get model list: {e}")
        except json.JSONDecodeError as e:
            print(f"Warning: Could not parse Ollama model list response: {e}")


        lmstudio_models = []

        try:
            response = requests.get("http://localhost:1234/v1/models", timeout=3)
            response.raise_for_status()
            models_data = response.json()
            
            if "data" in models_data and isinstance(models_data["data"], list):
                for model_info in models_data["data"]:
                    if "id" in model_info:
                        lmstudio_models.append(f"lmstudio:{model_info['id']}")

            if not lmstudio_models:
                print("Warning: No models found in LM Studio API response, though connection was successful.")

        except Exception as e:
            print(f"Error connecting to LM Studio: {e}")


        combined_models = static_models + ollama_models + lmstudio_models

        # Ollamaモデルが一つも取得できなかった場合、静的なOllamaモデルをフォールバックとして追加
        if not any(m.startswith("ollama:") for m in combined_models):
            print("Warning: No Ollama models fetched dynamically. Adding static Ollama models as fallback.")
            static_ollama_fallbacks = [
                "ollama:llama3",
                "ollama:qwen2.5vl:7b", # OllamaのVisionモデル例
                "ollama:gemma3:4b-it-qat",
                "ollama:llava", # OllamaのVisionモデル例
            ]
            combined_models.extend(static_ollama_fallbacks)

        return sorted(list(set(combined_models))) # 重複除去とソート
    
    def select_model(self, model_name: str):
        if None is model_name or "" == model_name :
            return

        index = self.model_combo.findText(model_name)
        #print("select_model model_name", model_name)
        #print("select_model index", index)
        if index < 0:
            return
        

#        for frame in inspect.stack():
#            print(frame.filename, frame.lineno, frame.function)

        if index != -1:
            self.model_combo.setCurrentIndex(index)
            self.on_model_selected(index)
        
    def get_selected_model(self) -> str:
        return self.model_combo.currentText()

    def set_font(self, font):
        self.model_label.setFont(font)
        self.model_combo.setFont(font)
        self.refresh_button.setFont(font)

    def set_tools_vision_text(self, text: str):
        self.tools_vision.setText(text)

    def on_model_selected(self, index):
        QApplication.setOverrideCursor(Qt.WaitCursor)
        name = self.model_combo.itemText(index)

#        for frame in inspect.stack():
#            print(frame.filename, frame.lineno, frame.function)
        
        #print("on_model_selected name", name)
        # ここに処理を書く
        if self.selected_function:
            
            self.selected_function(name)
        QApplication.restoreOverrideCursor()    
    def set_selected_function(self, func):
        self.selected_function = func
    
    def set_tools_vision_text(self, text: str):
        self.tools_vision.setText(text)
        
    def insert_widget(self, index, widget):
        self.settings_layout.insertWidget(index, widget)
    
    def set_sort_function(self, func):
        self.model_combo.set_sort_function(func)