#!/usr/bin/env python3
"""
ファイル: list_files_recursive.py
作成者: ChatGPT
変更修正者:robotcreation-dialy.com

説明:
指定されたフォルダに含まれるすべてのファイルパス（サブフォルダも含む）のリストを返す単一の関数 `list_files_recursive(folder_path)` を提供します。返されるパスは絶対パスで、アルファベット順に並べられています。
スクリプトを直接実行した際に、現在の作業ディレクトリで関数を実行する小さなテストハーネスが含まれています。
"""

import os
from typing import List
import json

class FileList():
    def __init__(self):
        self.g_file_description = {}
        self.g_data_file_path = ""

    def list_files_recursive(self, folder_path: str) -> List[str]:
        """
        *folder_path* 配下のすべてのファイルパスのリストを再帰的に返します。

        パラメータ
        ----------
        folder_path : str
        スキャンするディレクトリへのパス。相対パスまたは絶対パスで指定します。
        パスが存在しないかディレクトリでない場合は、ValueError がスローされます。

        戻り値
        -------
        List[str]
        ソートされた絶対ファイルパスのリスト。ディレクトリは除外されます。

        注記
        -----
        * 隠しファイル（ドットで始まるもの）も含まれます。
        * ファイルを指すシンボリックリンクも含まれます。ディレクトリを指すリンクは os.walk によって辿られます。
        * この関数は、大規模なディレクトリツリーに効率的な `os.walk` を使用します。
        """
        global g_file_description
        global g_data_file_path
        folder_path = folder_path.replace("\\", "/")
        if not os.path.isdir(folder_path):
            raise ValueError(f"'{folder_path}' is not a valid directory")

        abs_root = os.path.abspath(folder_path)
        file_list: List[str] = []

        for root, dirs, files in os.walk(abs_root):
            for name in files:
                file_path = os.path.join(root, name)
                file_list.append(file_path)

        file_list.sort()
        g_file_description = {}
        for file_path in file_list:
            file_path = file_path.replace("\\", "/")
            file_path = file_path.replace(folder_path, "")
            self.set_file_description(file_path)
        if os.path.exists(g_data_file_path):
            self.load_file_descriptions()
        else:
            self.save_file_descriptions()

        return file_list

    def create_folder(self, folder_path):
        """
        指定されたフォルダが存在しない場合、フォルダを作成します。

        Args:
          folder_path: 作成するフォルダのパス。
        """
        folder_path = folder_path.replace("\\", "/")
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)
        if folder_path.endswith("/"):
            return folder_path
        else:
            return folder_path + "/"
        
    def get_parent_directory(self, file_path):
        """
        fiel_pathからファイル名を除いてさらに親フォルダのパスを得る関数。

        Args:
            file_path: ファイルのパス。

        Returns:
            親フォルダのパス。ファイルが存在しない場合はNoneを返す。
        """
        if not os.path.exists(file_path):
            return None
        parent_dir = os.path.dirname(file_path)
        grandparent_dir = os.path.dirname(parent_dir)
        return grandparent_dir

    def get_directory(self, file_path):
        """
        fiel_pathからファイル名を除いてさらに親フォルダのパスを得る関数。

        Args:
            file_path: ファイルのパス。

        Returns:
            親フォルダのパス。ファイルが存在しない場合はNoneを返す。
        """
        if not os.path.exists(file_path):
            return None
        parent_dir = os.path.dirname(file_path)

        return parent_dir

    def set_data_file_path(self, file_path):
        """
        データファイルのパスを設定します。

        Args:
          file_path: データファイルのパス。
        """
        global g_data_file_path
        g_data_file_path = file_path

    def print_file_descriptions(self):
        """
        現在のファイルの説明をコンソールに出力します。
        """
        global g_file_description
        for file_path, description in g_file_description.items():
            print(f"{file_path}: {description}")


    def set_file_description(self, file_path, purpose_of_creation = "",description=""):
        """
        指定されたファイルの説明を作成します。

        Args:
          file_path: 説明を作成するファイルのパス。
          purpose_of_creation: ファイル作成目的
          description: ファイル内容の説明
        """
        global g_file_description
        g_file_description[file_path] = {"purpose_of_creation":purpose_of_creation,"description":description}

    def save_file_descriptions(self):
        """
        ファイルの説明を指定されたパスに保存します。

        Args:
          output_path: 説明を保存するファイルのパス。
        """
        global g_file_description
        global g_data_file_path
        with open(g_data_file_path, "w", encoding="utf-8") as f:
            json.dump(g_file_description, f, indent=4, ensure_ascii=False)

    def load_file_descriptions(self):
        """
        指定されたパスからファイルの説明を読み込みます。

        Args:
          input_path: 説明を読み込むファイルのパス。
        """
        global g_file_description
        global g_data_file_path
        if not os.path.exists(g_data_file_path):
            return
        with open(g_data_file_path, "r", encoding="utf-8") as f:
            g_file_description = json.load(f)
        self.clean_up_file_descriptions()

    def get_all_file_descriptions(self) -> str:
        """
        現在のすべてのファイルの説明を取得します。

        Returns:
          すべてのファイルの説明を含む辞書。
        """
        global g_file_description
        # g_file_descriptionをJSON文字列として返す
        result = json.dumps(g_file_description, indent=4, ensure_ascii=False)
        return result

    def clean_up_file_descriptions(self):
        """
        存在しないファイルを説明リストから削除します。
        """
        global g_file_description
        to_delete = []
        for file_path in g_file_description.keys():
            if not os.path.exists(file_path):
                to_delete.append(file_path)
        for file_path in to_delete:
            del g_file_description[file_path]

    def get_and_create_all_file_descriptions(self, folder_path: str) -> List[str]:
        """
        *folder_path* 配下のすべてのファイルパスのリストを再帰的に返します。

        パラメータ
        ----------
        folder_path : str
        スキャンするディレクトリへのパス。相対パスまたは絶対パスで指定します。
        パスが存在しないかディレクトリでない場合は、ValueError がスローされます。

        戻り値
        -------
        List[str]
        ソートされた絶対ファイルパスのリスト。ディレクトリは除外されます。

        注記
        -----
        * 隠しファイル（ドットで始まるもの）も含まれます。
        * ファイルを指すシンボリックリンクも含まれます。ディレクトリを指すリンクは os.walk によって辿られます。
        * この関数は、大規模なディレクトリツリーに効率的な `os.walk` を使用します。
        """
        self.create_folder(self, folder_path)
        file_path =  os.path.join(folder_path, "progress_file_list.json")
        self.set_data_file_path(file_path)
        if not os.path.exists(file_path):
            self.list_files_recursive(folder_path)

        else:

            self.load_file_descriptions()
            self.clean_up_file_descriptions()    
        return self.get_all_file_descriptions()
    
    def get_file_description(self,file_path, default_value):
        """
        指定されたファイルの説明を取得します。
    
        Args:
          file_path: 説明を取得するファイルのパス。
    
        Returns:
          ファイルの説明。説明が存在しない場合は空文字列を返す。
        """        
        return self.g_file_description(file_path, default_value)