import requests
import io
import os
import sounddevice as sd
from scipy.io.wavfile import read
import json

# AivisSpeech APIのエンドポイントを定義 (VOICEVOX互換を想定)
AIVIS_SPEECH_BASE_URL = os.getenv('AIVIS_SPEECH_BASE_URL', 'http://127.0.0.1:10101')

def get_available_speakers():
    """
    AivisSpeech APIから利用可能な話者とスタイルのリストを取得して表示します。
    成功した場合、話者の情報が含まれるリストを返します。
    """
    speakers_url = f"{AIVIS_SPEECH_BASE_URL}/speakers"
    try:
        print(f"'{speakers_url}' から利用可能な話者リストを取得中...")
        response = requests.get(speakers_url, timeout=10)
        response.raise_for_status()
        speakers_data = response.json()

        print("\n利用可能な話者とスタイル:")
        if not speakers_data:
            print("  話者が見つかりませんでした。")
            return None

        for speaker in speakers_data:
            speaker_name = speaker.get("name", "不明な話者")
            print(f"- {speaker_name}:")
            styles = speaker.get("styles", [])
            if not styles:
                print("    - スタイルが見つかりません。")
            for style in styles:
                style_name = style.get("name", "不明なスタイル")
                style_id = style.get("id", "N/A")
                print(f"    - {style_name} (ID: {style_id})")

        return speakers_data

    except requests.exceptions.RequestException as e:
        print(f"話者リストの取得中にエラーが発生しました: {e}")
    except Exception as e:
        print(f"話者リストの処理中に予期せぬエラーが発生しました: {e}")

    return None

def speak(text: str, speaker_id: int = 1):
    """
    指定されたテキストをAivisSpeech APIに送信して音声で読み上げます。
    (VOICEVOX互換の2段階POST方式)
    """
    try:
        # 1. audio_query (音声合成用のクエリを作成)
        audio_query_url = f"{AIVIS_SPEECH_BASE_URL}/audio_query"
        # VOICEVOX API仕様では、パラメータ名は 'speaker'
        params_query = {
            "text": text,
            "speaker": speaker_id
        }
        print(f"'{audio_query_url}' へクエリを送信中 (params={params_query})...")

        # パラメータをクエリ文字列としてPOSTリクエストを送信
        query_response = requests.post(audio_query_url, params=params_query, timeout=10)
        query_response.raise_for_status()
        audio_query_data = query_response.json()
        print("クエリの作成に成功しました。")

        # 2. synthesis (音声データを合成)
        synthesis_url = f"{AIVIS_SPEECH_BASE_URL}/synthesis"
        params_synthesis = {
            "speaker": speaker_id
        }
        synthesis_headers = {'Content-Type': 'application/json'}

        print(f"'{synthesis_url}' へ音声合成リクエストを送信中...")
        synthesis_response = requests.post(
            synthesis_url,
            params=params_synthesis,
            json=audio_query_data,
            headers=synthesis_headers,
            timeout=20
        )
        synthesis_response.raise_for_status()

        # レスポンスが音声データ(wav)か確認
        if 'audio/wav' not in synthesis_response.headers.get('Content-Type', ''):
            print(f"警告: AivisSpeech APIからの応答が音声データではありませんでした: {synthesis_response.headers.get('Content-Type')}")
            try:
                # エラーメッセージがJSONで返ってくる場合を想定
                error_text = synthesis_response.json()
                print(f"APIからのエラーメッセージ: {error_text}")
            except json.JSONDecodeError:
                print(f"APIからのテキスト応答: {synthesis_response.text}")
            return

        print("音声データの取得に成功しました。")

        # 3. 音声データを再生
        wav_data = synthesis_response.content

        # メモリ上のWAVデータを読み込む
        samplerate, data = read(io.BytesIO(wav_data))

        print("音声を再生します...")
        sd.play(data, samplerate)
        sd.wait()  # 再生が完了するまで待機

    except requests.exceptions.RequestException as e:
        print(f"AivisSpeech APIへの接続中にエラーが発生しました: {e}")
        if hasattr(e, 'response') and e.response is not None:
            print(f"  ステータスコード: {e.response.status_code}")
            print(f"  レスポンス: {e.response.text}")
        print(f"'{AIVIS_SPEECH_BASE_URL}' が起動しているか、ネットワーク接続を確認してください。")
    except Exception as e:
        print(f"音声の読み上げ中にエラーが発生しました: {e}")
