import uuid
from typing import List, Optional, Dict
from datetime import datetime


# ===============================
# ProgressNode クラス
# ===============================
# 個々のタスクやサブタスクを表現するノード。
# 階層構造を持ち、進捗状態・スケジュール・並列性・IDを管理する。
class ProgressNode:
    def __init__(
        self,
        name: str,                         # タスク名（例：「プロンプト設計」）
        description: str = "",             # タスクの説明（任意）
        status: str = "未着手",            # ステータス（未着手 / 進行中 / 完了 / 保留 / 中断）
        start_time: Optional[datetime] = None,  # タスクの開始時刻（15分単位で管理可能）
        end_time: Optional[datetime] = None,    # タスクの終了時刻
        can_run_parallel: bool = False     # 並列実行可能かどうか（Trueなら他タスクと同時進行可能）
    ):
        self.id = str(uuid.uuid4())  # 各ノードに一意なIDを自動生成（UUID形式）
        self.name = name
        self.description = description
        self.status = status
        self.start_time = start_time
        self.end_time = end_time
        self.can_run_parallel = can_run_parallel
        self.children: List["ProgressNode"] = []  # 子ノード（サブタスク）を格納するリスト

    # 子ノード（サブタスク）を追加
    def add_child(self, child: "ProgressNode"):
        self.children.append(child)

    # ステータスを更新（例：「進行中」→「完了」）
    def update_status(self, status: str):
        self.status = status

    # このノードが開始可能かどうかを判定
    def is_ready_to_start(self) -> bool:
        if not self.children:
            return True  # 子がなければすぐに開始可能
        if self.can_run_parallel:
            return True  # 並列実行可能なら子の完了を待たずに開始可能
        return all(child.status == "完了" for child in self.children)  # 子がすべて完了していればOK

    # ノード情報を辞書形式で出力（JSON化や保存用）
    def to_dict(self) -> Dict:
        return {
            "id": self.id,
            "name": self.name,
            "description": self.description,
            "status": self.status,
            "start_time": self.start_time.isoformat() if self.start_time else None,
            "end_time": self.end_time.isoformat() if self.end_time else None,
            "can_run_parallel": self.can_run_parallel,
            "children": [child.to_dict() for child in self.children]
        }

    # Mermaidガントチャート用の行を再帰的に生成
    def get_mermaid_lines(self) -> List[str]:
        lines = []
        if self.start_time and self.end_time:
            # Mermaid形式: タスク名 :ステータス, 開始時刻, 終了時刻
            line = f'{self.name} :{self.status}, {self.start_time.isoformat()}, {self.end_time.isoformat()}'
            lines.append(line)
        for child in self.children:
            lines.extend(child.get_mermaid_lines())
        return lines

    # IDを指定してノードを再帰的に検索
    def find_by_id(self, target_id: str) -> Optional["ProgressNode"]:
        if self.id == target_id:
            return self
        for child in self.children:
            result = child.find_by_id(target_id)
            if result:
                return result
        return None


# ===============================
# ProgressTracker クラス
# ===============================
# プロジェクト全体を管理するクラス。
# ルートノードを持ち、ID検索・ステータス更新・可視化出力などを提供。
class ProgressTracker:
    def __init__(self, title: str):
        self.title = title  # プロジェクト名（ガントチャートのタイトルなどに使用）
        self.root: Optional[ProgressNode] = None  # ルートノード（プロジェクト全体の起点）

    # プロジェクトのルートノードを設定
    def set_root(self, node: ProgressNode):
        self.root = node

    # プロジェクト全体をJSON形式で出力（AIや外部連携用）
    def to_json(self) -> Dict:
        return {
            "title": self.title,
            "root": self.root.to_dict() if self.root else {}
        }

    # Mermaid記法でガントチャートを出力（人間向け可視化）
    def to_mermaid(self) -> str:
        if not self.root:
            return "gantt\n    title 未設定\n"
        lines = [
            "gantt",
            f"    title {self.title}",
            "    dateFormat  YYYY-MM-DDTHH:mm:ss",  # ISO形式で15分単位の時間管理を可能に
            "    axisFormat  %H:%M"  # 時刻表示を「時:分」に設定
        ]
        lines.extend(["    " + line for line in self.root.get_mermaid_lines()])
        return "\n".join(lines)

    # ツリー構造をコンソールに表示（デバッグや構造確認用）
    def print_tree(self):
        def _print(node: ProgressNode, indent: int = 0):
            print("  " * indent + f"- {node.name} [{node.status}] (ID: {node.id})")
            for child in node.children:
                _print(child, indent + 1)
        if self.root:
            _print(self.root)

    # IDを指定してノードを検索（ルート以下を再帰的に探索）
    def find_node_by_id(self, node_id: str) -> Optional[ProgressNode]:
        if not self.root:
            return None
        return self.root.find_by_id(node_id)

    # IDを指定してステータスを更新（成功すればTrue、失敗すればFalse）
    def update_status_by_id(self, node_id: str, new_status: str) -> bool:
        node = self.find_node_by_id(node_id)
        if node:
            node.update_status(new_status)
            return True
        return False