import json
from typing import List, Dict, Any, Optional
from datetime import datetime

# 記憶の1単位を表すクラス。記憶の種類（エピソード、セマンティックなど）と内容、タイムスタンプを保持。
class MemoryEntity:
    def __init__(self, memory_type: str, content: Any, timestamp: Optional[str] = None):
        self.memory_type = memory_type  # 記憶の種類（例：'episodic', 'semantic', 'task'）
        self.content = content          # 記憶の内容（任意のデータ型）
        self.timestamp = timestamp or datetime.utcnow().isoformat()  # タイムスタンプ（デフォルトは現在時刻）

    # JSON形式に変換するための辞書化メソッド
    def to_dict(self) -> Dict[str, Any]:
        return {
            "type": self.memory_type,
            "content": self.content,
            "timestamp": self.timestamp
        }

# 認知ワークスペース全体を管理するクラス。複数の記憶層を持ち、追加・検索・エクスポートが可能。
class CognitiveMemory:
    def __init__(self):
        # 即時作業用の記憶（短期的な推論や思考に使用）
        self.scratchpad: List[MemoryEntity] = []

        # タスク固有の記憶（問題解決中の状態や中間結果など）
        self.task_buffer: List[MemoryEntity] = []

        # エピソード記憶（過去の出来事や体験の記録）
        self.episodic_cache: List[MemoryEntity] = []

        # セマンティック記憶（一般知識や意味的な情報）
        self.semantic_bridge: List[MemoryEntity] = []

    # 指定された層に記憶を追加する
    def add_memory(self, layer: str, memory: MemoryEntity):
        if layer == "scratchpad":
            self.scratchpad.append(memory)
        elif layer == "task":
            self.task_buffer.append(memory)
        elif layer == "episodic":
            self.episodic_cache.append(memory)
        elif layer == "semantic":
            self.semantic_bridge.append(memory)
        else:
            raise ValueError(f"未知の記憶層です: {layer}")

    # クエリ文字列に一致する記憶を検索。層を指定すればその層のみ、未指定なら全層から検索。
    def retrieve(self, query: str, layer: Optional[str] = None) -> List[MemoryEntity]:
        def match(entity: MemoryEntity) -> bool:
            # 記憶内容をJSON文字列化して検索語と照合（大文字小文字を無視）
            return query.lower() in json.dumps(entity.to_dict()).lower()

        # 層ごとの記憶リストを辞書化
        layers = {
            "scratchpad": self.scratchpad,
            "task": self.task_buffer,
            "episodic": self.episodic_cache,
            "semantic": self.semantic_bridge
        }

        # 指定層があればその層のみ検索、なければ全層を対象に検索
        if layer:
            return [e for e in layers[layer] if match(e)]
        else:
            results = []
            for mem_list in layers.values():
                results.extend([e for e in mem_list if match(e)])
            return results

    # 現在の記憶状態をJSON形式でエクスポート（保存や外部連携に利用可能）
    def export_json(self) -> str:
        return json.dumps({
            "scratchpad": [m.to_dict() for m in self.scratchpad],
            "task_buffer": [m.to_dict() for m in self.task_buffer],
            "episodic_cache": [m.to_dict() for m in self.episodic_cache],
            "semantic_bridge": [m.to_dict() for m in self.semantic_bridge]
        }, indent=2)

    # 指定された記憶層をクリア（記憶をすべて削除）
    def clear_layer(self, layer: str):
        if layer == "scratchpad":
            self.scratchpad.clear()
        elif layer == "task":
            self.task_buffer.clear()
        elif layer == "episodic":
            self.episodic_cache.clear()
        elif layer == "semantic":
            self.semantic_bridge.clear()
        else:
            raise ValueError(f"未知の記憶層です: {layer}")
        

    # 共通ユーティリティ：層名からリストを取得
    def _get_layer(self, layer: str) -> List[MemoryEntity]:
        if layer == "scratchpad":
            return self.scratchpad
        elif layer == "task":
            return self.task_buffer
        elif layer == "episodic":
            return self.episodic_cache
        elif layer == "semantic":
            return self.semantic_bridge
        else:
            raise ValueError(f"未知の記憶層です: {layer}")

    # 各層に記憶を追加
    def add_to_scratchpad(self, memory: MemoryEntity):
        self.scratchpad.append(memory)

    def add_to_task_buffer(self, memory: MemoryEntity):
        self.task_buffer.append(memory)

    def add_to_episodic_cache(self, memory: MemoryEntity):
        self.episodic_cache.append(memory)

    def add_to_semantic_bridge(self, memory: MemoryEntity):
        self.semantic_bridge.append(memory)

    # 各層から記憶を検索（クエリ文字列に一致するもの）
    def search_scratchpad(self, query: str) -> List[MemoryEntity]:
        return [m for m in self.scratchpad if query.lower() in json.dumps(m.to_dict()).lower()]

    def search_task_buffer(self, query: str) -> List[MemoryEntity]:
        return [m for m in self.task_buffer if query.lower() in json.dumps(m.to_dict()).lower()]

    def search_episodic_cache(self, query: str) -> List[MemoryEntity]:
        return [m for m in self.episodic_cache if query.lower() in json.dumps(m.to_dict()).lower()]

    def search_semantic_bridge(self, query: str) -> List[MemoryEntity]:
        return [m for m in self.semantic_bridge if query.lower() in json.dumps(m.to_dict()).lower()]

    # 各層の記憶をすべて削除
    def clear_scratchpad(self):
        self.scratchpad.clear()

    def clear_task_buffer(self):
        self.task_buffer.clear()

    def clear_episodic_cache(self):
        self.episodic_cache.clear()

    def clear_semantic_bridge(self):
        self.semantic_bridge.clear()

    # 各層の記憶を一覧表示（辞書形式）
    def list_scratchpad(self) -> List[Dict[str, Any]]:
        return [m.to_dict() for m in self.scratchpad]

    def list_task_buffer(self) -> List[Dict[str, Any]]:
        return [m.to_dict() for m in self.task_buffer]

    def list_episodic_cache(self) -> List[Dict[str, Any]]:
        return [m.to_dict() for m in self.episodic_cache]

    def list_semantic_bridge(self) -> List[Dict[str, Any]]:
        return [m.to_dict() for m in self.semantic_bridge]
