from abc import ABC, abstractmethod # LlmInterface.py
from typing import Any, List, Optional, Dict # Dictを追加
from langchain_core.language_models.chat_models import BaseChatModel
from langchain_core.messages import BaseMessage

class LLMInterface(ABC):
    model_name: str # 各実装でモデル名を保持

    @abstractmethod
    def __init__(self, model_identifier: str, temperature: float = 0, **kwargs):
        pass

    @abstractmethod
    def get_langchain_llm_instance(self) -> Optional[BaseChatModel]:
        """
        LangchainのBaseChatModel互換インスタンスを返す (AgentExecutor用)。
        AgentExecutorを使わない、または直接モデルを叩く場合はNoneを返しても良い。
        """
        pass

    @abstractmethod
    def get_response(self,
                     prompt: str,
                     chat_history: Optional[List[BaseMessage]] = None,
                     image_paths: Optional[List[str]] = None,
                     system_prompt: Optional[str] = None, # AIAgentから渡される
                     tools: Optional[List[Any]] = None,       # AIAgentから渡される
                     callbacks: Optional[List[Any]] = None, # AIAgentから渡される
                     **kwargs) -> str:
        """
        LLMからの応答を取得する。
        ツールが指定されていれば、このメソッド内でAgentExecutorを作成・利用することを想定。
        """
        pass

    # clear_memory はAIAgentがLangchainのMemoryオブジェクトを管理するため、LLMInterfaceからは削除。
    # LLMプロバイダ固有のキャッシュクリアが必要な場合は別途検討。

    @property
    @abstractmethod
    def supports_images(self) -> bool:
        pass

    # オプショナルなメソッドとして、ツールやシステムプロンプトの更新をインターフェースに追加
    def update_tools(self, tools: List[Any]):
        """LLMプロバイダが保持するツールリストを更新する (オプション)"""
        # デフォルト実装は何もしない。必要に応じてサブクラスでオーバーライド。
        pass

    def update_system_prompt(self, system_prompt: str):
        """LLMプロバイダが保持するシステムプロンプトを更新する (オプション)"""
        pass
